// normally the model is passed to the view instead of the view creating the model
define([
    'underscore',
    'backbone',
    'marionette',
    'moment',
    'Vent',
    'text!modules/new-appointment-request/views/direct/preferred-date/templates/preferred-date-layout.html',

    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-by-date-collection',
    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-collection',

    'modules/form/question-model',
    'modules/form/question-view',
    'text!modules/new-appointment-request/views/direct/preferred-date/templates/_preferred-date-description.html',
    'modules/new-appointment-request/views/direct/preferred-date/preferred-date-available',
    'modules/new-appointment-request/views/direct/preferred-date/closest-dates',
    'modules/new-appointment-request/views/direct/preferred-date/all-dates',
    'modules/new-appointment-request/views/direct/preferred-date/selected-date-time',
    'text!modules/new-appointment-request/views/direct/preferred-date/templates/_selected-date-time-aria-message.html',
],
function(
    _,
    Backbone,
    Marionette,
    moment,
    radio,
    template,
    TimeSlotsByDateCollection,
    TimeSlotsCollection,
    QuestionModel, QuestionView,
    _preferredDateDescriptionTemplate,
    PreferredDateAvailableView,
    ClosestDatesView,
    AllDatesView,
    SelectedDateTimeView,
    _ariaLiveMessageTemplate
) {
    'use strict';

    // when a button is clicked ('time-slot:selected'), save the date to this.model (form model)
    return Backbone.Marionette.View.extend({
        template: template,
        regions: {
            preferredDateRegion: '.form-section',
            availableDatesTimesRegion: '.available-dates-times-section',
            selectedDateTimeSlotRegion: '.selected-date-time-slot-section',
            ariaLiveRegion: '.content-live-region',
        },
        $activeButton: $(), // a TimeButtonView
        modelEvents: {
            'change:desiredDate': 'toggleAvailabilityButton resetAvailableDatesTimeRegion hideAllDatesButton',
        },
        events: {
            'click #show-availability-btn': 'showAvailability',
            'click #show-all-dates-btn': 'showAllDatesView',
        },
        initialize: function() {
            this.preferredDateModel = new QuestionModel({
                'id': 'desiredDate',
                'class': 'preferred-date-question',
                'type': 'text-datepicker',
                'label': 'Preferred Date:',
                'descriptionTemplate': _preferredDateDescriptionTemplate,
                'required': true,
                'validate': {
                    'rules': {
                        'formattedDate': true,
                        'dateInRange': true,
                    },
                },
            });
        },
        // given a sorted time slots collection, group time slots by date
        groupTimeSlotsByDate: function(timeSlotsCollection) {
            var groupedTimeSlots = new TimeSlotsByDateCollection();
            var dateLength = 10;
            
            timeSlotsCollection.each(function(timeSlot) {
                var dateString = timeSlot.get('startDateTime').substring(0, dateLength);
                var currentGroup = groupedTimeSlots.at(groupedTimeSlots.length - 1);
                if (_.isUndefined(currentGroup) || currentGroup.get('date') !== dateString) {
                    groupedTimeSlots.add({
                        date: dateString,
                        timeSlots: new TimeSlotsCollection([timeSlot]),
                    }, {silent: true});
                } else {
                    // add silently, eventing not needed yet
                    currentGroup.get('timeSlots').add(timeSlot, {silent: true});
                }
            });

            return groupedTimeSlots;
        },
        updateFDNS   elPreferredDate: function(preferredDateModel) {
            var dateString = '';
            var desiredDate = preferredDateModel.get('value');
            if (desiredDate !== '' && !_.isUndefined(desiredDate)) {
                desiredDate.setHours(0, 0, 0, 0);
                dateString = moment(preferredDateModel.get('value')).format('MM/DD/YYYY 00:00:00');
            }
            if (dateString !== '') {
                this.model.set('desiredDate', dateString);
            } else {
                this.model.unset('desiredDate');
            }
            if (this.model.has('dateTime')) {
                this.model.unset('dateTime');
            }
        },
        updateFDNS   elDateTime: function(timeSlotModel) {
            var oneMinInMs = 60000;
            var sTime = new Date(timeSlotModel.get('startDateTime'));
            var eTime = new Date(timeSlotModel.get('endDateTime'));
            this.model.set({
                dateTime: timeSlotModel.get('startDateTime'),
                apptLength: (eTime - sTime) / oneMinInMs,
            });
        },

        onRender: function() {
            this.showPreferredDateQuestionView();
            this.showSelectedDateTimeView();
            this.updateSelectedDateTimeAriaMessage();

            this.timeSlotsByDateCollection = this.groupTimeSlotsByDate(this.collection);

            this.toggleAvailabilityButton();
            this.hideAllDatesButton();

            radio.off('time-slot:selected', undefined, this);
            radio.on('time-slot:selected', function(timeButtonView, timeSlotModel) {
                this.updateFDNS   elDateTime(timeSlotModel);
                this.updateActiveButton(timeButtonView);
            }, this);
        },
        onDestroy: function() {
            radio.off('time-slot:selected', undefined, this);
            this.stopListening(this.preferredDateModel, 'change:value', undefined);
            delete this.preferredDateModel;
            delete this.$activeButton;
        },
        toggleAvailabilityButton: function() {
            if (this.model.has('desiredDate')) {
                this.$el.find('#show-availability-btn')
                    .prop('disabled', false)
                    .attr('aria-disabled', false)
                    .removeClass('ui-state-disabled')
                    .removeAttr('aria-label');
            } else {
                this.$el.find('#show-availability-btn')
                    .prop('disabled', true)
                    .attr('aria-disabled', true)
                    .addClass('ui-state-disabled')
                    .attr('aria-label', 'Enter a preferred date to enable Show Availability button');
            }
        },
        updateActiveButton: function(timeButtonView) {
            this.$activeButton.removeClass('ui-btn-active');
            this.$activeButton.attr('aria-label', this.$activeButton.text());

            this.$activeButton = timeButtonView.$el;
            this.$activeButton.addClass('ui-btn-active');
            this.$activeButton.attr('aria-label', this.$activeButton.text() + ' selected');
        },
        showPreferredDateQuestionView: function() {
            var preferredDateRegion = this.getRegion('preferredDateRegion');

            if (!this.isDestroyed()) {
                this.setMinMaxDates(this.preferredDateModel);
                
                preferredDateRegion.show(new QuestionView({model: this.preferredDateModel}));

                this.stopListening(this.preferredDateModel, 'change:value', undefined);
                this.listenTo(this.preferredDateModel, 'change:value', this.updateFDNS   elPreferredDate);
            }
        },
        setMinMaxDates: function(dateQuestionModel) {
            var today = new Date();
            var maxRange = 90;
            var minDate;
            var maxDate;
            
            today.setHours(0, 0, 0, 0);
            minDate = new Date(today);
            maxDate = new Date(today);
            minDate.setDate(minDate.getDate() + 1);
            maxDate.setDate(maxDate.getDate() + maxRange);

            dateQuestionModel.set({
                min: minDate,
                max: maxDate,
            });
        },
        showAvailability: function() {
            var dateLength = 10;
            if (typeof gas !== 'undefined') {
                gas('send', 'event', 'veteran-appointment', 'user-requested-show-availability');
            }

            if (this.timeSlotsByDateCollection.hasDate(this.model.get('desiredDate').substring(0, dateLength))) {
                this.showPreferredDateAvailableView();
                this.showAllDatesButton();
            } else {
                this.showClosestDatesView();
                this.showAllDatesButton();
            }

            if (typeof gas !== 'undefined') {
                gas('send', 'event', 'veteran-appointment', 'appointment-date-and-times-displayed');
            }
        },
        showPreferredDateAvailableView: function() {
            var availableDatesTimesRegion = this.getRegion('availableDatesTimesRegion');
            if (!this.isDestroyed()) {
                availableDatesTimesRegion.show(new PreferredDateAvailableView({
                    model: this.model,
                    timeSlotsByDateCollection: this.timeSlotsByDateCollection,
                }));
            }
        },
        showClosestDatesView: function() {
            var availableDatesTimesRegion = this.getRegion('availableDatesTimesRegion');
            if (!this.isDestroyed()) {
                availableDatesTimesRegion.show(new ClosestDatesView({
                    model: this.model,
                    timeSlotsByDateCollection: this.timeSlotsByDateCollection,
                }));
            }
        },
        showAllDatesView: function() {
            var availableDatesTimesRegion = this.getRegion('availableDatesTimesRegion');
            if (!this.isDestroyed()) {
                availableDatesTimesRegion.show(new AllDatesView({
                    model: this.model,
                    timeSlotsByDateCollection: this.timeSlotsByDateCollection,
                    customMessage: this.options.customMessage.get('messageText') || '',
                }));

                this.hideAllDatesButton();
            }
        },
        showAllDatesButton: function() {
            this.$el.find('#show-all-dates-btn').show();
        },
        hideAllDatesButton: function() {
            this.$el.find('#show-all-dates-btn').hide();
        },
        resetAvailableDatesTimeRegion: function() {
            var availableDatesTimesRegion = this.getRegion('availableDatesTimesRegion');

            availableDatesTimesRegion.reset();
            this.resetSelectedDateTimeSlotRegion();
        },
        resetSelectedDateTimeSlotRegion: function() {
            var selectedDateTimeSlotRegion = this.getRegion('selectedDateTimeSlotRegion');

            selectedDateTimeSlotRegion.reset();

            this.showSelectedDateTimeView();
        },
        showSelectedDateTimeView: function() {
            var selectedDateTimeSlotRegion = this.getRegion('selectedDateTimeSlotRegion');

            if (!this.isDestroyed()) {
                selectedDateTimeSlotRegion.show(new SelectedDateTimeView({model: this.model}));
            }
        },
        // NOTE: (for IE) message is a partial template instead of a view
        // because the wrapping element of the view interferes with IE + NVDA
        // reading the updated text
        updateSelectedDateTimeAriaMessage: function() {
            var ariaLiveRegion = this.getRegion('ariaLiveRegion');

            ariaLiveRegion.$el = this.$el.find(ariaLiveRegion.el);
            this.listenTo(this.model, 'change:dateTime', function() {
                if (!this.isDestroyed() && this.model.has('dateTime')) {
                    ariaLiveRegion.$el.empty().html(_.template(_ariaLiveMessageTemplate)(this.serializeData()));
                }
            });
        },
    });
});
